<?php
namespace Highrise\Modules\Shortcodes\ImageGallery;

use Highrise\Modules\Shortcodes\Lib\ShortcodeInterface;

class ImageGallery implements ShortcodeInterface
{

    private $base;

    /**
     * Image Gallery constructor.
     */
    public function __construct() {
        $this->base = 'mkd_image_gallery';

        add_action('vc_before_init', array($this, 'vcMap'));
    }

    /**
     * Returns base for shortcode
     * @return string
     */
    public function getBase() {
        return $this->base;
    }

    /**
     * Maps shortcode to Visual Composer. Hooked on vc_before_init
     *
     * @see mkd_core_get_carousel_slider_array_vc()
     */
    public function vcMap() {

        vc_map(array(
            'name' => esc_html__('Image Gallery', 'highrise'),
            'base' => $this->getBase(),
            'category' => esc_html__('by MIKADO', 'highrise'),
            'icon' => 'icon-wpb-image-gallery extended-custom-icon',
            'allowed_container_element' => 'vc_row',
            'params' => array(
                array(
                    'type' => 'attach_images',
                    'heading' => esc_html__('Images', 'highrise'),
                    'param_name' => 'images',
                    'description' => esc_html__('Select images from media library', 'highrise')
                ),
                array(
                    'type' => 'textfield',
                    'heading' => esc_html__('Image Size', 'highrise'),
                    'param_name' => 'image_size',
                    'description' => esc_html__('Enter image size. Example: thumbnail, medium, large, full or other sizes defined by current theme. Alternatively enter image size in pixels: 200x100 (Width x Height). Leave empty to use "thumbnail" size', 'highrise')
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Gallery Type', 'highrise'),
                    'admin_label' => true,
                    'param_name' => 'type',
                    'value' => array(
                        esc_html__('Slider', 'highrise') => 'slider',
                        esc_html__('Image Grid', 'highrise') => 'image_grid'
                    ),
                    'description' => esc_html__('Select gallery type', 'highrise'),
                    'save_always' => true
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Column Number', 'highrise'),
                    'param_name' => 'column_number',
                    'value' => array(2, 3, 4, 5),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'image_grid'
                        )
                    ),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Open PrettyPhoto on click', 'highrise'),
                    'param_name' => 'pretty_photo',
                    'value' => array(
                        esc_html__('No', 'highrise') => 'no',
                        esc_html__('Yes', 'highrise') => 'yes'
                    ),
                    'save_always' => true,
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Grayscale Images', 'highrise'),
                    'param_name' => 'grayscale',
                    'value' => array(
                        esc_html__('No', 'highrise') => 'no',
                        esc_html__('Yes', 'highrise') => 'yes'
                    ),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'image_grid'
                        )
                    )
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Slide duration', 'highrise'),
                    'admin_label' => true,
                    'param_name' => 'autoplay',
                    'value' => array(
                        esc_html__('3', 'highrise') => '3',
                        esc_html__('5', 'highrise') => '5',
                        esc_html__('10', 'highrise') => '10',
                        esc_html__('Disable', 'highrise') => 'disable'
                    ),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'slider'
                        )
                    ),
                    'description' => esc_html__('Auto rotate slides each X seconds', 'highrise'),
                    'group' => esc_html__('Slider', 'highrise'),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Pause on hover', 'highrise'),
                    'param_name' => 'pause',
                    'value' => array(
                        esc_html__('Yes', 'highrise') => 'yes',
                        esc_html__('No', 'highrise') => 'no'
                    ),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'autoplay',
                        'value' => array(
                            '3', '5', '10'
                        )
                    ),
                    'group' => esc_html__('Slider', 'highrise'),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Loop', 'highrise'),
                    'param_name' => 'loop',
                    'value' => array(
                        esc_html__('Yes', 'highrise') => 'yes',
                        esc_html__('No', 'highrise') => 'no'
                    ),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'slider'
                        )
                    ),
                    'group' => esc_html__('Slider', 'highrise'),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Show Navigation Arrows', 'highrise'),
                    'param_name' => 'navigation',
                    'value' => array(
                        esc_html__('Yes', 'highrise') => 'yes',
                        esc_html__('No', 'highrise') => 'no'
                    ),
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'slider'
                        )
                    ),
                    'save_always' => true,
                    'group' => esc_html__('Slider', 'highrise'),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Navigation Position', 'highrise'),
                    'param_name' => 'navigation_position',
                    'value' => array(
                        esc_html__('Inside', 'highrise') => 'inside',
                        esc_html__('Outside', 'highrise') => 'outside'
                    ),
                    'std' => 'inside',
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'navigation',
                        'value' => array(
                            'yes'
                        )
                    ),
                    'group' => esc_html__('Slider', 'highrise'),
                ),
                array(
                    'type' => 'dropdown',
                    'heading' => esc_html__('Show Pagination', 'highrise'),
                    'param_name' => 'pagination',
                    'value' => array(
                        esc_html__('Yes', 'highrise') => 'yes',
                        esc_html__('No', 'highrise') => 'no'
                    ),
                    'save_always' => true,
                    'dependency' => array(
                        'element' => 'type',
                        'value' => array(
                            'slider'
                        )
                    ),
                    'group' => esc_html__('Slider', 'highrise'),
                )
            )
        ));

    }

    /**
     * Renders shortcodes HTML
     *
     * @param $atts array of shortcode params
     * @param $content string shortcode content
     * @return string
     */
    public function render($atts, $content = null) {

        $args = array(
            'images' => '',
            'image_size' => 'thumbnail',
            'type' => 'slider',
            'pretty_photo' => '',
            'column_number' => '',
            'grayscale' => '',
            'autoplay' => '',
            'pause' => '',
            'loop' => '',
            'navigation' => '',
            'navigation_position' => '',
            'pagination' => '',
        );

        $params = shortcode_atts($args, $atts);
        $params['gallery_slider_data'] = $this->getGallerySliderData($params);
        $params['image_size'] = $this->getImageSize($params['image_size']);
        $params['images'] = $this->getGalleryImages($params);
        $params['pretty_photo'] = ($params['pretty_photo'] == 'yes') ? true : false;
        $params['columns'] = 'mkd-gallery-columns-' . $params['column_number'];
        $params['gallery_classes'] = $this->getGalleryClasses($params);

        if ($params['type'] == 'image_grid') {
            $template = 'gallery-grid';
        } elseif ($params['type'] == 'slider') {
            $template = 'gallery-slider';
        }

        $html = highrise_mikado_get_shortcode_module_template_part('templates/' . $template, 'image-gallery', '', $params);

        return $html;

    }

    private function getGalleryClasses($params) {
        $gallery_classes = array();

        $gallery_classes[] = ($params['grayscale'] == 'yes') ? 'mkd-grayscale' : '';
        $gallery_classes[] = ($params['navigation_position'] == 'inside') ? 'mkd-nav-inside' : '';

        return implode(' ', $gallery_classes);
    }

    /**
     * Return images for gallery
     *
     * @param $params
     * @return array
     */
    private function getGalleryImages($params) {
        $image_ids = array();
        $images = array();
        $i = 0;

        if ($params['images'] !== '') {
            $image_ids = explode(',', $params['images']);
        }

        foreach ($image_ids as $id) {

            $image['image_id'] = $id;
            $image_original = wp_get_attachment_image_src($id, 'full');
            $image['url'] = $image_original[0];
            $image['title'] = get_the_title($id);

            $images[$i] = $image;
            $i++;
        }

        return $images;

    }

    /**
     * Return image size or custom image size array
     *
     * @param $image_size
     * @return array
     */
    private function getImageSize($image_size) {

        $image_size = trim($image_size);
        //Find digits
        preg_match_all('/\d+/', $image_size, $matches);
        if (in_array($image_size, array('thumbnail', 'thumb', 'medium', 'large', 'full'))) {
            return $image_size;
        } elseif (!empty($matches[0])) {
            return array(
                $matches[0][0],
                $matches[0][1]
            );
        } else {
            return 'thumbnail';
        }
    }

    /**
     * Return all configuration data for slider
     *
     * @param $params
     * @return array
     */
    private function getGallerySliderData($params) {
        $gallery_slider_data = array();

        $gallery_slider_data['data-autoplay'] = ($params['autoplay'] !== '') ? $params['autoplay'] : '';
        $gallery_slider_data['data-pause'] = ($params['pause'] != '') ? $params['pause'] : '';
        $gallery_slider_data['data-loop'] = ($params['loop'] !== '') ? $params['loop'] : '';
        $gallery_slider_data['data-navigation'] = ($params['navigation'] !== '') ? $params['navigation'] : '';
        $gallery_slider_data['data-pagination'] = ($params['pagination'] !== '') ? $params['pagination'] : '';

        return $gallery_slider_data;
    }
}