<?php
namespace Highrise\Modules\Header\Types;

use Highrise\Modules\Header\Lib\HeaderType;

/**
 * Class that represents Header Standard layout and option
 *
 * Class HeaderStandard
 */
class HeaderStandard extends HeaderType
{
    protected $heightOfTransparency;
    protected $heightOfCompleteTransparency;
    protected $headerHeight;
    protected $mobileHeaderHeight;

    /**
     * Sets slug property which is the same as value of option in DB
     */
    public function __construct() {
        $this->slug = 'header-standard';

        if (!is_admin()) {

            $menuAreaHeight = highrise_mikado_filter_px(highrise_mikado_options()->getOptionValue('menu_area_height_header_standard'));
            $this->menuAreaHeight = $menuAreaHeight !== '' ? $menuAreaHeight : 100;

            $mobileHeaderHeight = highrise_mikado_filter_px(highrise_mikado_options()->getOptionValue('mobile_header_height'));
            $this->mobileHeaderHeight = $mobileHeaderHeight !== '' ? $mobileHeaderHeight : 100;

            add_action('wp', array($this, 'setHeaderHeightProps'));

            add_filter('highrise_mikado_js_global_variables', array($this, 'getGlobalJSVariables'));
            add_filter('highrise_mikado_per_page_js_vars', array($this, 'getPerPageJSVariables'));
            add_filter('highrise_mikado_add_page_custom_style', array($this, 'headerPerPageStyles'));
        }
    }

    public function headerPerPageStyles($style) {
        $id = highrise_mikado_get_page_id();
        $class_prefix = highrise_mikado_get_unique_page_class();
        $main_menu_style = array();
        $main_menu_grid_style = array();
        $disable_grid_shadow = highrise_mikado_get_meta_field_intersect('menu_area_in_grid_shadow_header_standard', $id) == 'no';
        $current_style = '';
        $main_menu_selector = array($class_prefix . '.mkd-header-standard .mkd-page-header .mkd-menu-area');
        $main_menu_grid_selector = array($class_prefix . '.mkd-header-standard .mkd-page-header .mkd-menu-area .mkd-grid .mkd-vertical-align-containers');

        /* header style - start */

        $menu_area_background_color = get_post_meta($id, 'mkd_menu_area_background_color_header_standard_meta', true);
        $menu_area_background_transparency = get_post_meta($id, 'mkd_menu_area_background_transparency_header_standard_meta', true);
        $menu_area_border = get_post_meta($id, 'mkd_menu_area_border_header_standard_meta', true);
        $menu_area_border_color = get_post_meta($id, 'mkd_menu_area_border_header_standard_color_meta', true);

        if ($menu_area_background_transparency === '') {
            $menu_area_background_transparency = 1;
        }

        $menu_area_background_color_rgba = highrise_mikado_rgba_color($menu_area_background_color, $menu_area_background_transparency);

        if ($menu_area_background_color_rgba !== null) {
            $main_menu_style['background-color'] = $menu_area_background_color_rgba;
        }

        if ($menu_area_border === 'no') {
            $main_menu_style['border-bottom'] = 'none';
        } elseif ($menu_area_border === 'yes' && $menu_area_border_color !== '') {
            $main_menu_style['border-bottom'] = '1px solid ' . $menu_area_border_color;
        }

        /* header style - end */

        /* header in grid style - start */

        if (!$disable_grid_shadow) {
            $main_menu_grid_style['box-shadow'] = '0px 1px 3px rgba(0,0,0,0.15)';
        }

        $menu_area_grid_background_color = get_post_meta($id, 'mkd_menu_area_grid_background_color_header_standard_meta', true);
        $menu_area_grid_background_transparency = get_post_meta($id, 'mkd_menu_area_grid_background_transparency_header_standard_meta', true);

        if ($menu_area_grid_background_transparency === '') {
            $menu_area_grid_background_transparency = 1;
        }

        $menu_area_grid_background_color_rgba = highrise_mikado_rgba_color($menu_area_grid_background_color, $menu_area_grid_background_transparency);

        if ($menu_area_grid_background_color_rgba !== null) {
            $main_menu_grid_style['background-color'] = $menu_area_grid_background_color_rgba;
        }

        /* header in grid style - end */

        $current_style .= highrise_mikado_dynamic_css($main_menu_selector, $main_menu_style);
        $current_style .= highrise_mikado_dynamic_css($main_menu_grid_selector, $main_menu_grid_style);

        $current_style = $current_style . $style;

        return $current_style;
    }

    /**
     * Loads template file for this header type
     *
     * @param array $parameters associative array of variables that needs to passed to template
     */
    public function loadTemplate($parameters = array()) {
        $id = highrise_mikado_get_page_id();

        $parameters['menu_area_in_grid'] = highrise_mikado_get_meta_field_intersect('menu_area_in_grid_header_standard', $id) == 'yes' ? true : false;
        $parameters['menu_area_position'] = highrise_mikado_get_meta_field_intersect('menu_area_position_header_standard', $id);

        $parameters['menu_area_position_class'] = '';
        if ($parameters['menu_area_position'] == 'right') {
            $parameters['menu_area_position_class'] = 'mkd-menu-area-position-right';
        }

        $parameters = apply_filters('highrise_mikado_header_standard_parameters', $parameters);

        highrise_mikado_get_module_template_part('templates/types/' . $this->slug, $this->moduleName, '', $parameters);
    }

    /**
     * Sets header height properties after WP object is set up
     */
    public function setHeaderHeightProps() {
        $this->heightOfTransparency = $this->calculateHeightOfTransparency();
        $this->heightOfCompleteTransparency = $this->calculateHeightOfCompleteTransparency();
        $this->headerHeight = $this->calculateHeaderHeight();
        $this->mobileHeaderHeight = $this->calculateMobileHeaderHeight();
    }

    /**
     * Returns total height of transparent parts of header
     *
     * @return int
     */
    public function calculateHeightOfTransparency() {
        $id = highrise_mikado_get_page_id();
        $transparencyHeight = 0;

        if (get_post_meta($id, 'mkd_menu_area_background_color_header_standard_meta', true) !== '') {
            $menuAreaTransparent = get_post_meta($id, 'mkd_menu_area_background_color_header_standard_meta', true) !== '' &&
                get_post_meta($id, 'mkd_menu_area_background_transparency_header_standard_meta', true) !== '1';
        } elseif (highrise_mikado_options()->getOptionValue('menu_area_background_color_header_standard') == '') {
            $menuAreaTransparent = highrise_mikado_options()->getOptionValue('menu_area_grid_background_color_header_standard') !== '' &&
                highrise_mikado_options()->getOptionValue('menu_area_grid_background_transparency_header_standard') !== '1';
        } else {
            $menuAreaTransparent = highrise_mikado_options()->getOptionValue('menu_area_background_color_header_standard') !== '' &&
                highrise_mikado_options()->getOptionValue('menu_area_background_transparency_header_standard') !== '1';
        }

        $sliderExists = get_post_meta($id, 'mkd_page_slider_meta', true) !== '';
        $contentBehindHeader = get_post_meta($id, 'mkd_page_content_behind_header_meta', true) === 'yes';

        if ($contentBehindHeader) {
            $menuAreaTransparent = true;
        }

        if ($menuAreaTransparent) {
            $transparencyHeight = $this->menuAreaHeight;

            if (($sliderExists && highrise_mikado_is_top_bar_enabled())
                || highrise_mikado_is_top_bar_enabled() && highrise_mikado_is_top_bar_transparent()
            ) {
                $transparencyHeight += highrise_mikado_get_top_bar_height();
            }
        }

        return $transparencyHeight;
    }

    /**
     * Returns height of completely transparent header parts
     *
     * @return int
     */
    public function calculateHeightOfCompleteTransparency() {
        $id = highrise_mikado_get_page_id();
        $transparencyHeight = 0;

        if (get_post_meta($id, 'mkd_menu_area_background_color_header_standard_meta', true) !== '') {
            $menuAreaTransparent = get_post_meta($id, 'mkd_menu_area_background_color_header_standard_meta', true) !== '' &&
                get_post_meta($id, 'mkd_menu_area_background_transparency_header_standard_meta', true) === '0';
        } elseif (highrise_mikado_options()->getOptionValue('menu_area_background_color_header_standard') == '') {
            $menuAreaTransparent = highrise_mikado_options()->getOptionValue('menu_area_grid_background_color_header_standard') !== '' &&
                highrise_mikado_options()->getOptionValue('menu_area_grid_background_transparency_header_standard') === '0';
        } else {
            $menuAreaTransparent = highrise_mikado_options()->getOptionValue('menu_area_background_color_header_standard') !== '' &&
                highrise_mikado_options()->getOptionValue('menu_area_background_transparency_header_standard') === '0';
        }

        if ($menuAreaTransparent) {
            $transparencyHeight = $this->menuAreaHeight;
        }

        return $transparencyHeight;
    }


    /**
     * Returns total height of header
     *
     * @return int|string
     */
    public function calculateHeaderHeight() {
        $headerHeight = $this->menuAreaHeight;
        if (highrise_mikado_is_top_bar_enabled()) {
            $headerHeight += highrise_mikado_get_top_bar_height();
        }

        return $headerHeight;
    }

    /**
     * Returns total height of mobile header
     *
     * @return int|string
     */
    public function calculateMobileHeaderHeight() {
        $mobileHeaderHeight = $this->mobileHeaderHeight;

        return $mobileHeaderHeight;
    }

    /**
     * Returns global js variables of header
     *
     * @param $globalVariables
     *
     * @return int|string
     */
    public function getGlobalJSVariables($globalVariables) {
        $globalVariables['mkdLogoAreaHeight'] = 0;
        $globalVariables['mkdMenuAreaHeight'] = $this->headerHeight;
        $globalVariables['mkdMobileHeaderHeight'] = $this->mobileHeaderHeight;

        return $globalVariables;
    }

    /**
     * Returns per page js variables of header
     *
     * @param $perPageVars
     *
     * @return int|string
     */
    public function getPerPageJSVariables($perPageVars) {
        //calculate transparency height only if header has no sticky behaviour
        if (!in_array(highrise_mikado_get_meta_field_intersect('header_behaviour'), array(
            'sticky-header-on-scroll-up',
            'sticky-header-on-scroll-down-up'
        ))
        ) {
            $perPageVars['mkdHeaderTransparencyHeight'] = $this->headerHeight - (highrise_mikado_get_top_bar_height() + $this->heightOfCompleteTransparency);
        } else {
            $perPageVars['mkdHeaderTransparencyHeight'] = 0;
        }

        return $perPageVars;
    }
}